import uncertainties
from scipy import integrate

class Integral:
    """Integral wrapper class

    Args:
        func (callable): function to integrate
        Xlow (float): lower integration bound
        Xup (float): upper integration bound

    Calling an instance of this class:
        params_unc (tuple): the parameters with (correlated) uncertainty
    
    Returns:
        tuple: integral, integral uncertainty
    """   
    def __init__(self, func, Xlow, Xup):       
        self.func = func
        self.bounds = Xlow, Xup
        self.__name__ = func.__name__
    
    def _integrate(self, *params):
        return integrate.quad(self.func, *self.bounds, args=params)[0]

    def eval(self, params):
        """Evaluate integral without uncertainties

        Args:
            params (tuple): function parameters

        Returns:
            float: value of the integral
        """        
        return self.integrate(*params)

    def __call__(self, params_unc):
        wrapper = uncertainties.wrap(self._integrate)
        result = wrapper(*params_unc)
        return result.nominal_value, result.std_dev
